﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/inspector2/Inspector2Request.h>
#include <aws/inspector2/Inspector2_EXPORTS.h>
#include <aws/inspector2/model/FilterAction.h>

#include <utility>

namespace Aws {
namespace Inspector2 {
namespace Model {

/**
 */
class ListFiltersRequest : public Inspector2Request {
 public:
  AWS_INSPECTOR2_API ListFiltersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListFilters"; }

  AWS_INSPECTOR2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The Amazon resource number (ARN) of the filter.</p>
   */
  inline const Aws::Vector<Aws::String>& GetArns() const { return m_arns; }
  inline bool ArnsHasBeenSet() const { return m_arnsHasBeenSet; }
  template <typename ArnsT = Aws::Vector<Aws::String>>
  void SetArns(ArnsT&& value) {
    m_arnsHasBeenSet = true;
    m_arns = std::forward<ArnsT>(value);
  }
  template <typename ArnsT = Aws::Vector<Aws::String>>
  ListFiltersRequest& WithArns(ArnsT&& value) {
    SetArns(std::forward<ArnsT>(value));
    return *this;
  }
  template <typename ArnsT = Aws::String>
  ListFiltersRequest& AddArns(ArnsT&& value) {
    m_arnsHasBeenSet = true;
    m_arns.emplace_back(std::forward<ArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The action the filter applies to matched findings.</p>
   */
  inline FilterAction GetAction() const { return m_action; }
  inline bool ActionHasBeenSet() const { return m_actionHasBeenSet; }
  inline void SetAction(FilterAction value) {
    m_actionHasBeenSet = true;
    m_action = value;
  }
  inline ListFiltersRequest& WithAction(FilterAction value) {
    SetAction(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token to use for paginating results that are returned in the response. Set
   * the value of this parameter to null for the first request to a list action. If
   * your response returns more than the <code>maxResults</code> maximum value it
   * will also return a <code>nextToken</code> value. For subsequent calls, use the
   * <code>nextToken</code> value returned from the previous request to continue
   * listing results after the first page.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListFiltersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results the response can return. If your request would
   * return more than the maximum the response will return a <code>nextToken</code>
   * value, use this value when you call the action again to get the remaining
   * results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListFiltersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_arns;
  bool m_arnsHasBeenSet = false;

  FilterAction m_action{FilterAction::NOT_SET};
  bool m_actionHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Inspector2
}  // namespace Aws
