﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cognito-idp/CognitoIdentityProviderRequest.h>
#include <aws/cognito-idp/CognitoIdentityProvider_EXPORTS.h>
#include <aws/cognito-idp/model/AttributeType.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace CognitoIdentityProvider {
namespace Model {

/**
 * <p>Represents the request to update the user's attributes as an
 * administrator.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cognito-idp-2016-04-18/AdminUpdateUserAttributesRequest">AWS
 * API Reference</a></p>
 */
class AdminUpdateUserAttributesRequest : public CognitoIdentityProviderRequest {
 public:
  AWS_COGNITOIDENTITYPROVIDER_API AdminUpdateUserAttributesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AdminUpdateUserAttributes"; }

  AWS_COGNITOIDENTITYPROVIDER_API Aws::String SerializePayload() const override;

  AWS_COGNITOIDENTITYPROVIDER_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the user pool where you want to update user attributes.</p>
   */
  inline const Aws::String& GetUserPoolId() const { return m_userPoolId; }
  inline bool UserPoolIdHasBeenSet() const { return m_userPoolIdHasBeenSet; }
  template <typename UserPoolIdT = Aws::String>
  void SetUserPoolId(UserPoolIdT&& value) {
    m_userPoolIdHasBeenSet = true;
    m_userPoolId = std::forward<UserPoolIdT>(value);
  }
  template <typename UserPoolIdT = Aws::String>
  AdminUpdateUserAttributesRequest& WithUserPoolId(UserPoolIdT&& value) {
    SetUserPoolId(std::forward<UserPoolIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the user that you want to query or modify. The value of this
   * parameter is typically your user's username, but it can be any of their alias
   * attributes. If <code>username</code> isn't an alias attribute in your user pool,
   * this value must be the <code>sub</code> of a local user or the username of a
   * user from a third-party IdP.</p>
   */
  inline const Aws::String& GetUsername() const { return m_username; }
  inline bool UsernameHasBeenSet() const { return m_usernameHasBeenSet; }
  template <typename UsernameT = Aws::String>
  void SetUsername(UsernameT&& value) {
    m_usernameHasBeenSet = true;
    m_username = std::forward<UsernameT>(value);
  }
  template <typename UsernameT = Aws::String>
  AdminUpdateUserAttributesRequest& WithUsername(UsernameT&& value) {
    SetUsername(std::forward<UsernameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of name-value pairs representing user attributes.</p> <p>For custom
   * attributes, you must prepend the <code>custom:</code> prefix to the attribute
   * name.</p> <p>If your user pool requires verification before Amazon Cognito
   * updates an attribute value that you specify in this request, Amazon Cognito
   * doesn’t immediately update the value of that attribute. After your user receives
   * and responds to a verification message to verify the new value, Amazon Cognito
   * updates the attribute value. Your user can sign in and receive messages with the
   * original attribute value until they verify the new value.</p> <p>To skip the
   * verification message and update the value of an attribute that requires
   * verification in the same API request, include the <code>email_verified</code> or
   * <code>phone_number_verified</code> attribute, with a value of <code>true</code>.
   * If you set the <code>email_verified</code> or <code>phone_number_verified</code>
   * value for an <code>email</code> or <code>phone_number</code> attribute that
   * requires verification to <code>true</code>, Amazon Cognito doesn’t send a
   * verification message to your user.</p>
   */
  inline const Aws::Vector<AttributeType>& GetUserAttributes() const { return m_userAttributes; }
  inline bool UserAttributesHasBeenSet() const { return m_userAttributesHasBeenSet; }
  template <typename UserAttributesT = Aws::Vector<AttributeType>>
  void SetUserAttributes(UserAttributesT&& value) {
    m_userAttributesHasBeenSet = true;
    m_userAttributes = std::forward<UserAttributesT>(value);
  }
  template <typename UserAttributesT = Aws::Vector<AttributeType>>
  AdminUpdateUserAttributesRequest& WithUserAttributes(UserAttributesT&& value) {
    SetUserAttributes(std::forward<UserAttributesT>(value));
    return *this;
  }
  template <typename UserAttributesT = AttributeType>
  AdminUpdateUserAttributesRequest& AddUserAttributes(UserAttributesT&& value) {
    m_userAttributesHasBeenSet = true;
    m_userAttributes.emplace_back(std::forward<UserAttributesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map of custom key-value pairs that you can provide as input for any custom
   * workflows that this action triggers.</p> <p>You create custom workflows by
   * assigning Lambda functions to user pool triggers. When you use the
   * AdminUpdateUserAttributes API action, Amazon Cognito invokes the function that
   * is assigned to the <i>custom message</i> trigger. When Amazon Cognito invokes
   * this function, it passes a JSON payload, which the function receives as input.
   * This payload contains a <code>clientMetadata</code> attribute, which provides
   * the data that you assigned to the ClientMetadata parameter in your
   * AdminUpdateUserAttributes request. In your function code in Lambda, you can
   * process the <code>clientMetadata</code> value to enhance your workflow for your
   * specific needs.</p> <p>For more information, see <a
   * href="https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html">
   * Using Lambda triggers</a> in the <i>Amazon Cognito Developer Guide</i>.</p>
   *  <p>When you use the <code>ClientMetadata</code> parameter, note that
   * Amazon Cognito won't do the following:</p> <ul> <li> <p>Store the
   * <code>ClientMetadata</code> value. This data is available only to Lambda
   * triggers that are assigned to a user pool to support custom workflows. If your
   * user pool configuration doesn't include triggers, the
   * <code>ClientMetadata</code> parameter serves no purpose.</p> </li> <li>
   * <p>Validate the <code>ClientMetadata</code> value.</p> </li> <li> <p>Encrypt the
   * <code>ClientMetadata</code> value. Don't send sensitive information in this
   * parameter.</p> </li> </ul>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetClientMetadata() const { return m_clientMetadata; }
  inline bool ClientMetadataHasBeenSet() const { return m_clientMetadataHasBeenSet; }
  template <typename ClientMetadataT = Aws::Map<Aws::String, Aws::String>>
  void SetClientMetadata(ClientMetadataT&& value) {
    m_clientMetadataHasBeenSet = true;
    m_clientMetadata = std::forward<ClientMetadataT>(value);
  }
  template <typename ClientMetadataT = Aws::Map<Aws::String, Aws::String>>
  AdminUpdateUserAttributesRequest& WithClientMetadata(ClientMetadataT&& value) {
    SetClientMetadata(std::forward<ClientMetadataT>(value));
    return *this;
  }
  template <typename ClientMetadataKeyT = Aws::String, typename ClientMetadataValueT = Aws::String>
  AdminUpdateUserAttributesRequest& AddClientMetadata(ClientMetadataKeyT&& key, ClientMetadataValueT&& value) {
    m_clientMetadataHasBeenSet = true;
    m_clientMetadata.emplace(std::forward<ClientMetadataKeyT>(key), std::forward<ClientMetadataValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_userPoolId;
  bool m_userPoolIdHasBeenSet = false;

  Aws::String m_username;
  bool m_usernameHasBeenSet = false;

  Aws::Vector<AttributeType> m_userAttributes;
  bool m_userAttributesHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_clientMetadata;
  bool m_clientMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace CognitoIdentityProvider
}  // namespace Aws
